/* @#[TAB_NAME="Cable Engineer"] */
/* @#[FULL_WINDOW='1']           */
import QtCharts
import QtCore
import QtQuick
import QtQuick.Layouts
import QtQuick.Controls.Basic
import "../UiLibrary"
import "../UiLibrary/formatUtil.mjs" as FU
import "Base"

PluginPanel {
	id: main_panel
	
	component ResetDistButton: Button { // Buttons for resetting section or total deviation
		palette.button: hovered ? PanelStyle.valueColor : buttonColor; palette.buttonText: (hovered & PanelStyle.inDarkMode) ? 'black' : 'white'
		background.implicitHeight: 20
		property string statusTip: 'Click to reset deviation to zero'
	}
	component LeftGauges: Item { // Tension and Ship & Cable Speed Gauges on LHS
		RowLayout {
			transform: Translate { x: -26; y: 0 }
			anchors.fill: parent; anchors.rightMargin: 13
			ColumnLayout {
				Layout.fillHeight: true
				Item {
					Layout.fillWidth: true;	height: childrenRect.height
					ItemLabel {
						text: `<b>Tension</b><br>[${forceMetric.unitName}]`; font.bold: false
						anchors.horizontalCenter: parent.horizontalCenter
						onContextMenu: forceUnitSelectHandler()
						statusTip: "Click right to change force units"
					}
				}
				Item {
					Layout.fillWidth: true;	Layout.fillHeight: true; Layout.bottomMargin: 21
					NeedleGauge {
						id: tension_gauge
						onContextMenu: forceUnitSelectHandler()
						valueBarWidth: gaugeBarWidth
						orientation: Qt.Vertical
						minimumValue: 0
						markerDecimals: 1
						barColor: Qt.rgba(1.0, 1.0, 1.0, 0.2)
						fillFromZero: true
						value: forceMetric.convert(fU.num(Cable.Primary.Smoothed.Tension, 0), DisplayMetric.FORCE.IN_kN, forceMetric.modeIndex)
						markerValue: forceMetric.convert(fU.num(Cable.Primary.Tension, 0), DisplayMetric.FORCE.IN_kN, forceMetric.modeIndex)
						maximumValue: fU.calcTickScale(0, tensionRangeMax(plugin.tensionScale, forceMetric.modeIndex)).maxValue
						//tickmarkStepSize: fU.calcTickScale(minimumValue, maximumValue).tickStep
						height: parent.height; implicitHeight: parent.height; anchors.horizontalCenter: parent.horizontalCenter
						readonly property real desiredTension: forceMetric.convert(plugin.desiredTensionValue, DisplayMetric.FORCE.IN_kN, forceMetric.modeIndex)
						readonly property real ttPercent: 1.0 - (desiredTension / maximumValue)
						readonly property bool ttValid: (ttPercent >= 0.1 && ttPercent < 0.95)
						gradient: Gradient {
							GradientStop { position: 0.0; color: gaugeRed }
							GradientStop { position: 0.05; color: gaugeRed }
							GradientStop { position: 0.1; color: gaugeYellow }
							GradientStop { position: tension_gauge.ttValid ? tension_gauge.ttPercent + 0.05 : 0.45; color: gaugeYellow }
							GradientStop { position: tension_gauge.ttValid ? tension_gauge.ttPercent : 0.5; color: tension_gauge.ttValid ? gaugeGreen : gaugeYellow }
							GradientStop { position: tension_gauge.ttValid ? tension_gauge.ttPercent - 0.05 : 0.55; color: gaugeYellow }
							GradientStop { position: 1.0; color: gaugeYellow }
						}
					}
				}
			}
			ColumnLayout {
				Item {
					Layout.fillWidth: true;	height: childrenRect.height
					ItemLabel {
						text: `<b>Ship Speed</b><br>[${speedMetric.unitName}]`;	font.bold: false
						anchors.horizontalCenter: parent.horizontalCenter
						onContextMenu: speedUnitSelectHandler()
						statusTip: "Click right to change speed units"
					}
				}
				Item {
					Layout.fillWidth: true;	Layout.fillHeight: true; Layout.bottomMargin: 25
					NeedleGauge {
						function calcStep() {
							var step = speedRange(plugin.speedScale, speedMetric.modeIndex)[2];
							if (step == 0) {
								step = fU.calcTickScale(minimumValue, maximumValue).tickStep;
								console.log(`Calculated stepsize is: ${step}`);
							}
							return step;
						}
						id: ship_speed_gauge
						onContextMenu: speedUnitSelectHandler()
						valueBarWidth: gaugeBarWidth
						orientation: Qt.Vertical
						barColor: Qt.rgba(1.0, 1.0, 1.0, 0.2)
						fillFromZero: true
						minimumValue: speedRange(plugin.speedScale, speedMetric.modeIndex)[0]
						value: speedMetric.convert(fU.num(Ship.SpeedKmh, 0), DisplayMetric.SPEED.IN_Km_h, speedMetric.modeIndex)
						markerValue: speedMetric.convert(fU.num(Ship.RawSpeedKmh, 0), DisplayMetric.SPEED.IN_Km_h, speedMetric.modeIndex)
						mark3Value: speedMetric.convert(fU.num(Dolog.SOG.MPS, 0), DisplayMetric.SPEED.IN_M_per_Sec, speedMetric.modeIndex)
						mark3Visible: fU.asString(Dolog.SOG.MPS) != ""
						maximumValue: speedRange(plugin.speedScale, speedMetric.modeIndex)[1]
						//tickmarkStepSize: calcStep() 
						height: parent.height; implicitHeight: parent.height; anchors.horizontalCenter: parent.horizontalCenter
						readonly property real desiredSpeed: speedMetric.convert(fU.num(Ship.DesiredSpeedKmh, 0), DisplayMetric.SPEED.IN_Km_h, speedMetric.modeIndex)
						readonly property real dsPercent: valueRatio(desiredSpeed)
						readonly property real dsPos: dsPercent >= 0 && dsPercent <= 1.0 ? dsPercent : -1.0
						readonly property real dsClamped: (dsPercent < 0.0) ? 0.0 : (dsPercent > 1.0 ? 1.0 : dsPercent)
						gradient: Gradient {
							GradientStop { position: 0.0; color: gaugeYellow }
							GradientStop { position: ship_speed_gauge.dsPos > 0.05 ? ship_speed_gauge.dsPos - 0.05 : 0.0; color: gaugeYellow }
							GradientStop { position: ship_speed_gauge.dsClamped; color: gaugeGreen }
							GradientStop { position: (ship_speed_gauge.dsClamped >= 0.0 && ship_speed_gauge.dsClamped < 0.95) ? ship_speed_gauge.dsClamped + 0.05 : 1.0; color: gaugeYellow }
							GradientStop { position: 1.0; color: gaugeYellow }
						}
						formatValue: function(v) {
							return v.toFixed(1);
						}	
					}
				}
			}
			ColumnLayout {
				Item {
					Layout.fillWidth: true;	height: childrenRect.height
					ItemLabel {
						text: `<b>Cable Speed</b><br>[${speedMetric.unitName}]`; font.bold: false
						anchors.horizontalCenter: parent.horizontalCenter
						onContextMenu: speedUnitSelectHandler()
						statusTip: "Click right to change speed units"
					}
				}
				Item {
					Layout.fillWidth: true; Layout.fillHeight: true; Layout.bottomMargin: 25
					NeedleGauge {
						function calcStep() {
							var step = speedRange(plugin.speedScale, speedMetric.modeIndex)[2];
							if (step == 0) {
								step = fU.calcTickScale(minimumValue, maximumValue).tickStep;
								console.log(`Calculated stepsize is: ${step}`);
							}
							return step;
						}
						id: cable_speed_gauge
						onContextMenu: speedUnitSelectHandler()
						valueBarWidth: gaugeBarWidth
						orientation: Qt.Vertical
						barColor: Qt.rgba(1.0, 1.0, 1.0, 0.2)
						fillFromZero: true
						minimumValue: speedRange(plugin.speedScale, speedMetric.modeIndex)[0]
						value: speedMetric.convert(fU.num(Cable.Primary.Smoothed.Speed, 0), DisplayMetric.SPEED.IN_Km_h, speedMetric.modeIndex)
						markerValue: speedMetric.convert(fU.num(Cable.Primary.Speed, 0), DisplayMetric.SPEED.IN_Km_h, speedMetric.modeIndex)
						maximumValue: speedRange(plugin.speedScale, speedMetric.modeIndex)[1]
						//tickmarkStepSize: calcStep()
						height: parent.height; implicitHeight: parent.height; anchors.horizontalCenter: parent.horizontalCenter
						readonly property real desiredSpeed: speedMetric.convert(fU.num(Cable.TargetSpeedKmh, 0), DisplayMetric.SPEED.IN_Km_h, speedMetric.modeIndex)
						readonly property real dsPercent: valueRatio(desiredSpeed)
						readonly property real dsPos: dsPercent >= 0 && dsPercent <= 1.0 ? dsPercent : -1.0
						readonly property real dsClamped: (dsPercent < 0.0) ? 0.0 : (dsPercent > 1.0 ? 1.0 : dsPercent)
						gradient: Gradient {
							GradientStop { position: 0.0; color: gaugeYellow }
							GradientStop { position: cable_speed_gauge.dsPos > 0.05 ? cable_speed_gauge.dsPos - 0.05 : 0.0; color: gaugeYellow }
							GradientStop { position: cable_speed_gauge.dsClamped; color: gaugeGreen }
							GradientStop { position: (cable_speed_gauge.dsClamped >= 0.0 && cable_speed_gauge.dsClamped < 0.95) ? cable_speed_gauge.dsClamped + 0.05 : 1.0; color: gaugeYellow }
							GradientStop { position: 1.0; color: gaugeYellow }
						}
						formatValue: function(v) {
							return v.toFixed(1);
						}	
					}
				}
			}		
		}	
	} // END LeftGauges
		
	component DeviationGauges: Item { // Distance and Slack deviation gauges top right
		id: toppan
		property bool swapSdGauge: settings.swapSdGauge
		readonly property int sdgDir: swapSdGauge ? -1 : 1
		ColumnLayout {
			anchors.fill: parent; anchors.topMargin: 59; spacing: 14; anchors.bottomMargin: 6
			RowLayout {
				id: hgLayout
				width: parent.width
				property real labelWidth: Math.max(slackdev_gauge_label.requiredWidth, distdev_gauge_label.requiredWidth)
				Item {
					id: slackdev_gauge_label
					property real requiredWidth: childrenRect.width
					Layout.fillHeight: true; width: hgLayout.labelWidth
					ItemLabel {	text: '<b>Slack</b><br>Deviation<br>%';	font.bold: false }
				}
				Item {
					Layout.fillWidth: true;	Layout.fillHeight: true; Layout.rightMargin: 30
					NeedleGauge {
						id: slackdev_gauge
						valueBarWidth: gaugeBarWidth
						barColor: Qt.rgba(1.0, 1.0, 1.0, 0.2)
						fillFromZero: true
						orientation: Qt.Horizontal
						minimumValue: sdgDir * -slackDeviationScale(plugin.slackDeviationScale)
						value: fU.num(Cable.SlackDeviation, 0)
						markerValue: fU.num(Cable.SlackDeviation, 0)
						maximumValue: sdgDir * slackDeviationScale(plugin.slackDeviationScale)
						//tickmarkStepSize: fU.calcTickScale(minimumValue, maximumValue).tickStep
						width: parent.width; anchors.horizontalCenter: parent.horizontalCenter
						gradient: Gradient {
							orientation: Gradient.Horizontal
							GradientStop { position: 0.0; color: gaugeRed }
							GradientStop { position: 0.05; color: gaugeYellow }
							GradientStop { position: 0.1; color: gaugeYellow }
							GradientStop { position: 0.45; color: gaugeYellow }
							GradientStop { position: 0.50; color: gaugeGreen }
							GradientStop { position: 0.55; color: gaugeYellow }
							GradientStop { position: 0.95; color: gaugeYellow }						
							GradientStop { position: 1.0; color: gaugeRed }
						}
						formatValue: function(value) { return value.toFixed(0) }
					}
				}
			}
			RowLayout {
				id: ddp
				property real value: fU.num(Cable.SectDistanceDeviation, 0)
				Item {
					id: distdev_gauge_label
					Layout.fillHeight: true
					property real requiredWidth: childrenRect.width
					width: hgLayout.labelWidth
					ItemLabel { text: '<b>Distance</b><br>Deviation<br>[m]'; font.bold: false }
				}
				Item {
					Layout.fillWidth: true;	Layout.fillHeight: true; Layout.rightMargin: 30
					NeedleGauge {
						id: distdev_gauge
						valueBarWidth: gaugeBarWidth
						barColor: Qt.rgba(1.0, 1.0, 1.0, 0.2)
						fillFromZero: true
						orientation: Qt.Horizontal
						minimumValue: -distanceDeviationScale(plugin.distanceDeviationScale)
						value: ddp.value
						markerValue: fU.num(Cable.RawSectDistanceDeviation, 0)
						maximumValue: distanceDeviationScale(plugin.distanceDeviationScale)
						//tickmarkStepSize: fU.calcTickScale(minimumValue, maximumValue).tickStep
						width: parent.width; anchors.horizontalCenter: parent.horizontalCenter
						gradient: Gradient {
							orientation: Gradient.Horizontal
							GradientStop { position: 0.0; color: gaugeRed }
							GradientStop { position: 0.05; color: gaugeYellow }
							GradientStop { position: 0.1; color: gaugeYellow }
							GradientStop { position: 0.45; color: gaugeYellow }
							GradientStop { position: 0.50; color: gaugeGreen }
							GradientStop { position: 0.55; color: gaugeYellow }
							GradientStop { position: 0.95; color: gaugeYellow }						
							GradientStop { position: 1.0; color: gaugeRed }
						}
						formatValue: function(value) { return value.toFixed(0) }
						ItemLabel {
							id: ddwarn
							property string message: (ddp.value > 5) ? 'Ship is >5m Ahead. Increase Payout' : ((ddp.value < -5) ? 'Ship is >5m Astern. Reduce Payout' : '')
							y: distdev_gauge.height; anchors.horizontalCenter: distdev_gauge.horizontalCenter
							color: PanelStyle.redValueColor; text: message; visible: message.length > 0
							OpacityAnimator on opacity {
								from: 0.4; to: 1.0;
								loops: Animation.Infinite; duration: 1000; running: ddwarn.visible
							}					
						}
					}
				}
			}
		}
	} // END DeviationGauges	

	component ChartPanel: Item { // Chart Area and series selection checkboxes
		property var checkModel: [
			{ series: smo_distdev },  { series: smo_slackdev },  { series: smo_cable_speed }, { series: smo_ship_speed }, { series: smo_tension },
			{ series: raw_distdev },  { series: raw_slackdev },  { series: raw_cable_speed }, { series: raw_ship_speed }, { series: raw_tension }  
		]
		property int totalPoints: 100		
		property int norecurse: 1			
		ColumnLayout {
			anchors.fill: parent; spacing: 0								
			Item {
				Layout.fillWidth: true;	Layout.fillHeight: true; Layout.margins: -10
				ChartView {
					id: chartview
					width: parent.width; height: parent.height
					legend.alignment: Qt.AlignBottom; antialiasing: true
					backgroundColor: PanelStyle.backgroundColor; titleColor: PanelStyle.labelColor
					legend.labelColor: PanelStyle.labelColor; legend.visible: false
					titleFont.pixelSize: 2
					ValueAxis { // time
						id: axisX
						min: 0;	max: totalPoints - 1; reverse: true						
						labelsColor: PanelStyle.inDarkMode ? "white" : "black"; gridLineColor: PanelStyle.inDarkMode ? "white" : "black"
						color: PanelStyle.inDarkMode ? "white" : "black"
					}
					ValueAxis { // tension
						id: axisLY1
						min: 0;	max: tensionRangeMax(plugin.tensionScale, forceMetric.modeIndex)
						labelsColor: PanelStyle.inDarkMode ? "#65FA20" : '#319203';	gridLineColor: PanelStyle.inDarkMode ? "white" : "black"
						color: PanelStyle.inDarkMode ? "white" : "black"
					}
					ValueAxis { // speed
						id: axisLY2
						min: speedRange(plugin.speedScale, speedMetric.modeIndex)[0]; max: speedRange(plugin.speedScale, speedMetric.modeIndex)[1]
						labelsColor: '#FA2020';	gridLineColor: PanelStyle.inDarkMode ? "white" : "black"
						color: PanelStyle.inDarkMode ? "white" : "black"; gridVisible: false
					}
					ValueAxis { // distance deviation
						id: axisRY1
						min: -distanceDeviationScale(plugin.distanceDeviationScale); max: distanceDeviationScale(plugin.distanceDeviationScale)
						labelsColor: '#2053FA';	gridLineColor: PanelStyle.inDarkMode ? "white" : "black"
						color: PanelStyle.inDarkMode ? "white" : "black"; gridVisible: false
					}
					ValueAxis { // slack deviation
						id: axisRY2
						min: -slackDeviationScale(plugin.slackDeviationScale); max: slackDeviationScale(plugin.slackDeviationScale)
						labelsColor: '#DA1FE0';	gridLineColor: PanelStyle.inDarkMode ? "white" : "black"
						color: PanelStyle.inDarkMode ? "white" : "black"; gridVisible: false
					}
					LineSeries {
						id: raw_tension; name: "Tension (raw)"; property int sindex: 6;	color: '#65FA20'; width: 2
						axisX: axisX; axisY: axisLY1
					}
					LineSeries {
						id: smo_tension; name: "Tension"; property int sindex: 1; color: '#C0FFA3'; width: 4
						axisX: axisX; axisY: axisLY1
					}
					LineSeries {
						id: raw_ship_speed; name: "Ship Speed (raw)"; property int sindex: 5; color: '#FA2020'; width: 2
						axisX: axisX; axisY: axisLY2
					}
					LineSeries {
						id: smo_ship_speed; name: "Ship Speed";	property int sindex: 0;	color: '#FF9C9C'; width: 4
						axisX: axisX; axisY: axisLY2
					}
					LineSeries {
						id: raw_cable_speed; name: "Cable Speed (raw)";	property int sindex: 9;	color: PanelStyle.inDarkMode ? '#E4D700' : '#c4cb05'; width: 2
						axisX: axisX; axisY: axisLY2
					}
					LineSeries {
						id: smo_cable_speed; name: "Cable Speed"; property int sindex: 4; color: PanelStyle.inDarkMode ? '#FDFFB3' : '#fbff3f'; width: 4
						axisX: axisX; axisY: axisLY2
					}
					LineSeries {
						id: raw_distdev; name: "Dist. Dev (raw)"; property int sindex: 7; color: '#2053FA'; width: 2
						axisX: axisX; axisYRight: axisRY1
					}
					LineSeries {
						id: smo_distdev; name: "Dist. Dev."; property int sindex: 2; color: '#9EB1F0'; width: 4
						axisX: axisX; axisYRight: axisRY1
					}
					LineSeries {
						id: raw_slackdev; name: "Slack Dev (raw)"; property int sindex: 8; color: '#DA1FE0'; width: 2
						axisX: axisX; axisYRight: axisRY2
					}
					LineSeries {
						id: smo_slackdev; name: "Slack Dev."; property int sindex: 3; color: '#E6A5E8'; width: 4
						axisX: axisX; axisYRight: axisRY2
					}
					Rectangle {
						x: chartview.plotArea.right - 66; y: chartview.plotArea.bottom - 34
						RowLayout {
							ZoomButton { labelText: '+'; statusTip: 'Click to zoom in';	 onClicked: zoom(10);  enabled: totalPoints < 100	}
							ZoomButton { labelText: '-'; statusTip: 'Click to zoom out'; onClicked: zoom(-10); enabled: totalPoints > 10	}
						}
					}
					Rectangle {
						id: bods
						x: chartview.plotArea.right - (childrenRect.width + 10); y: chartview.plotArea.top + 10						
						opacity: 0.5
						property real toNextClamped: fU.num(Cable.DistToNextBody, 99999)
						visible: toNextClamped > -500 && toNextClamped < 99999
						OpacityAnimator on opacity {
							from: 0.4; to: 0.85;
							loops: Animation.Infinite; duration: 1500							
							running: bods.toNextClamped > 0 && bods.toNextClamped <= 200
						}
						ColumnLayout {
							ItemLabel { text: 'Cable to go until body [km]' }
							ValueText { text: fU.formatScaledValue(Cable.DistToNextBody, 0.001, 1, 3) }
							ValueText { text: fU.asString(Cable.NextBodyName) }
							ValueText { text: bods.toNextClamped }
						}
					}
				}
			}
			Rectangle {				
				Layout.fillWidth: true; height: childrenRect.height
				color: PanelStyle.backgroundColor
				Layout.topMargin: -9; Layout.bottomMargin: 9
				GridLayout {
					columns: 5;	rowSpacing: 0; width: parent.width
					transform: Translate { x: 20 }
					Repeater {
						id: checkBoxes
						model: checkModel
						Item {
							Layout.fillWidth: true;	Layout.preferredHeight: childrenRect.height
							ColorCheckBox {
								anchors.horizontalCenter: parent.horizontalCenter
								text: checkBoxLabel(modelData.series.name)
								selColor: modelData.series.color; checked: modelData.series.visible
								onCheckedChanged: {
									if (norecurse == 0) { 
										modelData.series.visible = checked;
										updateSeriesVisibility();
									}
								}
							}
						}
					}				
				}
			}
		}
		Timer { id: refreshTimer; interval: 500; repeat: true; onTriggered: refreshData() }
				
		function checkBoxLabel(txt) { return (txt.indexOf('raw') >= 0) ? 'Raw' : txt; }	
		function updateSeriesVisibility() {
			norecurse++;
			var vflags = 0;
			for (var i = 0; i < checkModel.length; i++) {
				var series = checkModel[i].series;
				if (series.visible) {
					vflags |= 1 << series.sindex; 
				}
			}
			if (vflags == 0) {
				// we JUST wont allow you to deselect all checkboxes!!
				vflags = 1 << checkModel[3].series.sindex;
				settings.visibleSeries = String(vflags);
				restoreSeriesVisibility(vflags);
			}
			else {
				settings.visibleSeries = String(vflags);
			}
			norecurse--;
		}	
		function restoreSeriesVisibility(vflags = 0) {
			norecurse++;
			if (vflags == 0) {
				vflags = parseInt(settings.visibleSeries);
			}
			if (vflags > 0 && vflags < 1024) {
				for (var i = 0; i < checkModel.length; i++) {
					var series = checkModel[i].series;
					series.visible =  (vflags & (1 << series.sindex)) != 0;
				}
			}		
			norecurse--;
		}	
		function getRealData(key, multiplier) {
			var a = [];
			a.length = totalPoints;
			var y = plugin.seriesData(key).data;
			for (var i = 0; i < totalPoints; i++) {
				a[i] = Qt.point(totalPoints - (i + 1),y[i] * multiplier);
			}
			return a;
		}	
		function zoom(amount) {
			var q = Math.min(Math.max(10, totalPoints + amount), 100);
			if (totalPoints != q) {
				totalPoints = q;
				refreshData();
			}
		}								
		function refreshData() {
			VUtil.updateChartSeries(smo_ship_speed, getRealData(0, speedMetric.multiplierForConversion(DisplayMetric.SPEED.IN_Km_h)));
			VUtil.updateChartSeries(raw_ship_speed, getRealData(5, speedMetric.multiplierForConversion(DisplayMetric.SPEED.IN_Km_h)));
			VUtil.updateChartSeries(smo_cable_speed, getRealData(4, speedMetric.multiplierForConversion(DisplayMetric.SPEED.IN_Km_h)));
			VUtil.updateChartSeries(raw_cable_speed, getRealData(9, speedMetric.multiplierForConversion(DisplayMetric.SPEED.IN_Km_h)));
			VUtil.updateChartSeries(smo_tension, getRealData(1, forceMetric.multiplierForConversion(DisplayMetric.FORCE.IN_Kilonewton)));
			VUtil.updateChartSeries(raw_tension, getRealData(6, forceMetric.multiplierForConversion(DisplayMetric.FORCE.IN_Kilonewton)));
			VUtil.updateChartSeries(smo_distdev, getRealData(2, 1.0));
			VUtil.updateChartSeries(raw_distdev, getRealData(7, 1.0));
			VUtil.updateChartSeries(smo_slackdev, getRealData(3, 1.0));
			VUtil.updateChartSeries(raw_slackdev, getRealData(8, 1.0));
		}	
		function applyBackgroundGradient() {
			// Due to QML chartview design we cant seem to directly get a pointer to the QChart object from the QML item but we can obtain it in C++ from any series belonging to the chartview
			// so we pass the first series and let the C++ code get a pointer to the QChart
			VUtil.setChartBackgroundGradient(chartview.series(0), PanelStyle.inDarkMode ? [Qt.rgba(0.269, 0.269, 0.269, 1.0), PanelStyle.backgroundColor] : [PanelStyle.backgroundColor, Qt.darker(PanelStyle.backgroundColor, 1.2)]);
		}					
		function start() {
			applyBackgroundGradient();
			refreshData();
			restoreSeriesVisibility();
			refreshTimer.start();
			norecurse = 0;
			_wktnav.themeChanged.connect(function(name) {
				applyBackgroundGradient();
			});			
		}		
	} // END ChartPanel
	
	RowLayout {	// MAIN LAYOUT 
		anchors.fill: parent
		Item {
			Layout.preferredWidth: 410;	Layout.fillHeight: true
			ColumnLayout {
				anchors.fill: parent
				spacing: 0
				Item {
					Layout.preferredHeight: 133; Layout.fillWidth: true; Layout.topMargin: 8
					PanelBox {
						anchors.fill: parent; anchors.margins: 4
						text: 'Distance Deviation'
						GridLayout {
							columns: 2; width: parent.width; y: 0; rowSpacing: 2
							StdHyperlink { Layout.fillWidth: true; href: 'bsp://ddmethdialog'; label: `${plugin.distanceDeviationMethodName}`; statusTip: 'Click to change distance deviation method' }							
							ItemLabel    { Layout.fillWidth: true;text: `<font size="+2" color="${VUtil.colorName(PanelStyle.valueColor)}">${plugin.targetSlackValue.toFixed(2)}%</font> slack`          }
							ItemLabel { text: 'Section [m]'; Layout.fillWidth: true }
							ItemLabel { text: 'Total [km]';  Layout.fillWidth: true	}
							ValueText {	text: FU.formatValue(Cable.SectDistanceDeviation, 1, 3)	}
							ValueText {	text: FU.formatValue(Cable.CumlDistanceDeviation, 1, 3)	}
							ResetDistButton { text: '<b>Reset</b> (section)'; onClicked: { Cable.SectDistanceDeviation = 0; } }
							ResetDistButton { text: '<b>Reset</b> (total)';	  onClicked: { Cable.CumlDistanceDeviation = 0; } }
						}
					}
				}
				Item {
					Layout.minimumHeight: 210; Layout.fillWidth: true;	Layout.fillHeight: true
					LeftGauges { id: left_gauges; anchors.fill: parent }
				}
				Item {
					Layout.preferredHeight: 125; Layout.fillWidth: true; Layout.topMargin: 8
					PanelBox {
						anchors.fill: parent; anchors.margins: 5
						text: 'Slack'
						GridLayout {
							columns: 2; width: parent.width; y: 3; rowSpacing: 2
							ItemLabel {	text: 'Instantaneous %'; Layout.fillWidth: true	}							
							ItemLabel {	text: 'Target %';  		 Layout.fillWidth: true	}				
							ValueText {	text: FU.formatValue(Cable.InstantSlack, 1, 2) }
							ValueText {	text: plugin.targetSlackValue.toFixed(2) }
							ItemLabel { text: 'From last event %';	Layout.fillWidth: true	}							
							ItemLabel { text: 'Since last change %'; Layout.fillWidth: true	}						
							ValueText {	text: FU.formatValue(Cable.SlackFromSectStart, 1, 2) }
							ValueText {	text: FU.formatValue(Cable.SlackFromChangeOrLast200m, 1, 2)	}														
						}
					}					
				}
				Item {
					Layout.preferredHeight: 125; Layout.fillWidth: true; Layout.topMargin: 8
					PanelBox {
						anchors.fill: parent; anchors.margins: 5
						text: 'Distances'
						GridLayout {
							columns: 4;	width: parent.width; y: 3; rowSpacing: 2
							ItemLabel {	}							
							ItemLabel {	text: 'Cable [km]';	Layout.fillWidth: true; font.pixelSize: 13 }				
							ItemLabel {	text: 'Route [km]';	Layout.fillWidth: true; font.pixelSize: 13	}				
							ItemLabel {	text: 'Tot. Cable [km]'; Layout.fillWidth: true; font.pixelSize: 13	}				
							ItemLabel {	text: '<font size="-2">From<br>last pos</font>'; Layout.fillWidth: true	}
							ValueText { text: FU.formatValue(Cable.LaidInSection, 1, 3) }
							ValueText {	text: FU.formatScaledValue(Cable.RBDistFromStart, 0.001, 1, 3) }
							ValueText {	text: FU.formatValue(Cable.TotalLaid, 1, 3)	}
							ItemLabel {	Layout.columnSpan: 3 }
							ItemLabel { text: 'Total dist [km]'; font.pixelSize: 13	}
							ItemLabel {	text: '<font size="-2">To next<br>pos</font>'; Layout.fillWidth: true }
							ValueText {	text: FU.formatValue(Cable.InHand, 1, 3) }							
							ValueText { text: FU.formatScaledValue(Cable.RBDistToEnd, 0.001, 1, 3)	}
							ValueText {	text: FU.formatValue(Cable.TotalRouteDist, 1, 3) }							
						}
					}					
				}			
			}
		}
		Item {
			Layout.fillWidth: true;	Layout.fillHeight: true			
			ColumnLayout {
				anchors.fill: parent
				Item {
					Layout.fillWidth: true;	Layout.minimumHeight: 250; Layout.topMargin: 15
					DeviationGauges { id: dev_gauges; anchors.fill: parent 	}
				}
				Item {
					Layout.topMargin: -28
					Layout.fillHeight: true; Layout.fillWidth: true
					ChartPanel { id: chart_panel; anchors.fill: parent }
				}
				Item {
					id: botbox
					readonly property string endEventName: FU.asString(Cable.CLPEnd.EventName)
					Layout.fillWidth: true;	Layout.minimumHeight: 60
					PanelBox {
						anchors.fill: parent; anchors.margins: 5
						RowLayout {
							width: parent.width
							RowLayout { ItemLabel { text: 'NAV KP' } 	        ValueText { text: FU.formatValue(Ship.SP1.Route.KP, 1, 3) } }
							RowLayout {	ItemLabel { text: 'Water Depth [m]' }	ValueText { text: FU.formatValue(Ship.WaterDepth, 1, 1) } }
							RowLayout {	ItemLabel { text: 'End Event' }         ValueText { text: botbox.endEventName.length > 0 ? botbox.endEventName : '(none)' } }
							RowLayout {	ItemLabel { text: 'CLP No' }	        ValueText { text: FU.formatValue(Cable.CLPEnd.InstNum, 1, 0) } }							
						}
					}					
				}
			}
		}
	}
	MenuBarInsertion {
		path: 'View'
		Action { id: swapSdGauge; text: 'Reverse slack deviation gauge'; checkable: true; checked: toppan.swapSdGauge; onTriggered: settings.swapSdGauge = swapSdGauge.checked; shortcut: 'Ctrl+Shift+S' }
	}
	Component.onCompleted: {
		if (startPlugin()) {
			chart_panel.start();
		}
	}	
}